<?php
/**
 * Adobe Connect Main Class
 *
 * @package SafeFontsAdobeConnect
 * @since 1.0.0
 */

namespace Chrmrtns\SafeFontsAdobeConnect;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main Adobe Connect class
 */
class AdobeConnect {

    /**
     * Single instance
     *
     * @var AdobeConnect
     */
    private static $instance = null;

    /**
     * Adobe TypeKit API URL
     *
     * @var string
     */
    const TYPEKIT_API_URL = 'https://typekit.com/api/v1/json/kits/%s/published';

    /**
     * Adobe TypeKit CDN URL
     *
     * @var string
     */
    const TYPEKIT_CDN_URL = 'https://use.typekit.net/%s.css';

    /**
     * Option keys
     */
    const OPTION_PROJECT_ID = 'sfac_project_id';
    const OPTION_FONTS = 'sfac_adobe_fonts';

    /**
     * Get single instance
     *
     * @return AdobeConnect
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Admin
        add_action('admin_menu', array($this, 'add_admin_submenu'), 20);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_sfac_fetch_fonts', array($this, 'ajax_fetch_fonts'));
        add_action('wp_ajax_sfac_save_fonts', array($this, 'ajax_save_fonts'));
        add_action('wp_ajax_sfac_disconnect', array($this, 'ajax_disconnect'));

        // Frontend - only load if we have fonts configured
        if ($this->has_fonts()) {
            add_action('wp_enqueue_scripts', array($this, 'enqueue_adobe_fonts'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_adobe_fonts'));
            add_action('enqueue_block_editor_assets', array($this, 'enqueue_adobe_fonts'));

            // Register with SafeFonts Pro integrations
            add_filter('safefonts_pro_external_fonts', array($this, 'register_fonts'));

            // Page builder integrations
            $this->init_builder_integrations();
        }
    }

    /**
     * Check if we have Adobe fonts configured
     *
     * @return bool
     */
    public function has_fonts() {
        $project_id = get_option(self::OPTION_PROJECT_ID);
        $fonts = get_option(self::OPTION_FONTS);
        return !empty($project_id) && !empty($fonts);
    }

    /**
     * Get configured fonts
     *
     * @return array
     */
    public function get_fonts() {
        return get_option(self::OPTION_FONTS, array());
    }

    /**
     * Get project ID
     *
     * @return string
     */
    public function get_project_id() {
        return get_option(self::OPTION_PROJECT_ID, '');
    }

    /**
     * Add admin submenu under SafeFonts Pro
     */
    public function add_admin_submenu() {
        add_submenu_page(
            'safefonts-pro',
            __('Adobe Fonts', 'safefonts-adobe-connect'),
            __('Adobe Fonts', 'safefonts-adobe-connect') . ' <span class="sfac-badge">CDN</span>',
            'manage_options',
            'safefonts-adobe-connect',
            array($this, 'render_admin_page')
        );
    }

    /**
     * Enqueue admin assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_admin_assets($hook) {
        if ($hook !== 'safefonts-pro_page_safefonts-adobe-connect') {
            return;
        }

        wp_enqueue_style(
            'sfac-admin',
            CHRMRTNS_SFAC_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            CHRMRTNS_SFAC_VERSION
        );

        wp_enqueue_script(
            'sfac-admin',
            CHRMRTNS_SFAC_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            CHRMRTNS_SFAC_VERSION,
            true
        );

        wp_localize_script('sfac-admin', 'sfacAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sfac_nonce'),
            'strings' => array(
                'fetching' => __('Fetching fonts...', 'safefonts-adobe-connect'),
                'saving' => __('Saving...', 'safefonts-adobe-connect'),
                'error' => __('An error occurred. Please try again.', 'safefonts-adobe-connect'),
                'invalidId' => __('Please enter a valid 7-character Web Project ID.', 'safefonts-adobe-connect'),
                'confirmDisconnect' => __('Are you sure you want to disconnect Adobe Fonts? This will remove all Adobe fonts from your site.', 'safefonts-adobe-connect'),
            ),
        ));
    }

    /**
     * Render admin page
     */
    public function render_admin_page() {
        include CHRMRTNS_SFAC_PLUGIN_DIR . 'views/admin-page.php';
    }

    /**
     * Enqueue Adobe Fonts CSS from CDN
     */
    public function enqueue_adobe_fonts() {
        $project_id = $this->get_project_id();
        if (empty($project_id)) {
            return;
        }

        wp_enqueue_style(
            'safefonts-adobe-fonts',
            sprintf(self::TYPEKIT_CDN_URL, $project_id),
            array(),
            CHRMRTNS_SFAC_VERSION
        );
    }

    /**
     * AJAX: Fetch fonts from Adobe API
     */
    public function ajax_fetch_fonts() {
        check_ajax_referer('sfac_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied.', 'safefonts-adobe-connect'));
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $project_id = isset($_POST['project_id']) ? sanitize_text_field(wp_unslash($_POST['project_id'])) : '';

        if (empty($project_id) || strlen($project_id) !== 7) {
            wp_send_json_error(__('Invalid Web Project ID. Must be 7 characters.', 'safefonts-adobe-connect'));
        }

        $fonts = $this->fetch_fonts_from_api($project_id);

        if (is_wp_error($fonts)) {
            wp_send_json_error($fonts->get_error_message());
        }

        wp_send_json_success(array(
            'fonts' => $fonts,
            'project_id' => $project_id,
        ));
    }

    /**
     * Fetch fonts from Adobe TypeKit API
     *
     * @param string $project_id Web Project ID
     * @return array|WP_Error
     */
    private function fetch_fonts_from_api($project_id) {
        $api_url = sprintf(self::TYPEKIT_API_URL, $project_id);

        $response = wp_remote_get($api_url, array(
            'timeout' => 30,
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return new \WP_Error(
                'api_error',
                __('Could not fetch fonts. Please check your Web Project ID.', 'safefonts-adobe-connect')
            );
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (empty($body['kit']['families'])) {
            return new \WP_Error(
                'no_fonts',
                __('No fonts found in this Web Project.', 'safefonts-adobe-connect')
            );
        }

        $fonts = array();
        foreach ($body['kit']['families'] as $family) {
            $family_name = str_replace(' ', '-', $family['name']);
            $weights = array();

            foreach ($family['variations'] as $variation) {
                $chars = str_split($variation);
                $weight = $chars[1] . '00';
                if (!in_array($weight, $weights)) {
                    $weights[] = $weight;
                }
            }

            sort($weights);

            $fonts[$family_name] = array(
                'name' => $family['name'],
                'family' => $family_name,
                'slug' => $family['slug'],
                'css_name' => isset($family['css_names'][0]) ? $family['css_names'][0] : $family['slug'],
                'fallback' => str_replace('"', '', $family['css_stack']),
                'weights' => $weights,
            );
        }

        return $fonts;
    }

    /**
     * AJAX: Save selected fonts
     */
    public function ajax_save_fonts() {
        check_ajax_referer('sfac_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied.', 'safefonts-adobe-connect'));
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $project_id = isset($_POST['project_id']) ? sanitize_text_field(wp_unslash($_POST['project_id'])) : '';
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
        $fonts = isset($_POST['fonts']) ? map_deep(wp_unslash($_POST['fonts']), 'sanitize_text_field') : array();

        if (empty($project_id)) {
            wp_send_json_error(__('Missing project ID.', 'safefonts-adobe-connect'));
        }

        // Sanitize fonts array
        $sanitized_fonts = array();
        if (is_array($fonts)) {
            foreach ($fonts as $key => $font) {
                $sanitized_fonts[sanitize_text_field($key)] = array(
                    'name' => isset($font['name']) ? $font['name'] : '',
                    'family' => isset($font['family']) ? $font['family'] : '',
                    'slug' => isset($font['slug']) ? $font['slug'] : '',
                    'css_name' => isset($font['css_name']) ? $font['css_name'] : '',
                    'fallback' => isset($font['fallback']) ? $font['fallback'] : '',
                    'weights' => isset($font['weights']) && is_array($font['weights']) ? $font['weights'] : array(),
                );
            }
        }

        update_option(self::OPTION_PROJECT_ID, $project_id);
        update_option(self::OPTION_FONTS, $sanitized_fonts);

        wp_send_json_success(__('Adobe Fonts connected successfully!', 'safefonts-adobe-connect'));
    }

    /**
     * AJAX: Disconnect Adobe Fonts
     */
    public function ajax_disconnect() {
        check_ajax_referer('sfac_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied.', 'safefonts-adobe-connect'));
        }

        delete_option(self::OPTION_PROJECT_ID);
        delete_option(self::OPTION_FONTS);

        wp_send_json_success(__('Adobe Fonts disconnected.', 'safefonts-adobe-connect'));
    }

    /**
     * Register fonts with SafeFonts Pro
     *
     * @param array $fonts Existing external fonts
     * @return array
     */
    public function register_fonts($fonts) {
        $adobe_fonts = $this->get_fonts();

        foreach ($adobe_fonts as $font) {
            $fonts[] = array(
                'name' => $font['name'],
                'family' => $font['css_name'],
                'slug' => 'adobe-' . $font['slug'],
                'weights' => $font['weights'],
                'source' => 'adobe',
            );
        }

        return $fonts;
    }

    /**
     * Initialize page builder integrations
     */
    private function init_builder_integrations() {
        // Elementor
        add_filter('elementor/fonts/groups', array($this, 'elementor_font_group'));
        add_filter('elementor/fonts/additional_fonts', array($this, 'elementor_add_fonts'));

        // Bricks
        add_filter('bricks/builder/standard_fonts', array($this, 'bricks_add_fonts'));

        // Beaver Builder
        add_filter('fl_theme_system_fonts', array($this, 'beaver_add_fonts'));
        add_filter('fl_builder_font_families_system', array($this, 'beaver_add_fonts'));

        // Gutenberg / Block Editor
        add_action('enqueue_block_editor_assets', array($this, 'gutenberg_font_styles'));

        // Kadence Theme & Blocks
        add_filter('kadence_theme_add_custom_fonts', array($this, 'kadence_add_fonts'));
        add_filter('kadence_blocks_add_custom_fonts', array($this, 'kadence_add_fonts'));

        // Astra
        add_action('astra_customizer_font_list', array($this, 'astra_add_fonts'));

        // Note: GeneratePress/GenerateBlocks, Spectra, Blocksy require complex integrations
        // Not implemented in this version

        // Divi
        add_filter('et_websafe_fonts', array($this, 'divi_add_fonts'));

        // Etch - inject Adobe CSS into builder iframe
        add_filter('etch/preview/additional_stylesheets', array($this, 'etch_add_stylesheet'));

        // Note: Oxygen/Breakdance requires complex API integration
        // Not implemented in this version
    }

    /**
     * Elementor: Add font group
     */
    public function elementor_font_group($groups) {
        $groups['safefonts-adobe'] = __('Adobe Fonts (CDN)', 'safefonts-adobe-connect');
        return $groups;
    }

    /**
     * Elementor: Add fonts
     */
    public function elementor_add_fonts($fonts) {
        foreach ($this->get_fonts() as $font) {
            $fonts[$font['css_name']] = 'safefonts-adobe';
        }
        return $fonts;
    }

    /**
     * Bricks: Add fonts
     */
    public function bricks_add_fonts($fonts) {
        foreach ($this->get_fonts() as $font) {
            $fonts[] = $font['css_name'];
        }
        return $fonts;
    }

    /**
     * Beaver Builder: Add fonts
     */
    public function beaver_add_fonts($fonts) {
        foreach ($this->get_fonts() as $font) {
            $fonts[$font['css_name']] = array(
                'fallback' => $font['fallback'],
                'weights' => $font['weights'],
            );
        }
        return $fonts;
    }

    /**
     * Gutenberg: Add font styles
     */
    public function gutenberg_font_styles() {
        $project_id = $this->get_project_id();
        if (!empty($project_id)) {
            add_editor_style(sprintf(self::TYPEKIT_CDN_URL, $project_id));
        }
    }

    /**
     * Kadence: Add fonts
     */
    public function kadence_add_fonts($fonts) {
        foreach ($this->get_fonts() as $font) {
            $fonts[$font['css_name']] = array(
                'fallback' => $font['fallback'],
                'weights' => $font['weights'],
            );
        }
        return $fonts;
    }

    /**
     * Astra: Add fonts (outputs HTML options)
     *
     * @param string $value Currently selected font value
     */
    public function astra_add_fonts($value = '') {
        $fonts = $this->get_fonts();

        if (empty($fonts)) {
            return;
        }

        echo '<optgroup label="' . esc_attr__('Adobe Fonts (CDN)', 'safefonts-adobe-connect') . '">';

        foreach ($fonts as $font) {
            printf(
                '<option value="%s">%s</option>',
                esc_attr("'" . $font['css_name'] . "', " . $font['fallback']),
                esc_html($font['name'])
            );
        }

        echo '</optgroup>';
    }

    /**
     * Divi: Add fonts
     */
    public function divi_add_fonts($fonts) {
        foreach ($this->get_fonts() as $font) {
            $fonts[$font['css_name']] = array(
                'styles' => implode(',', $font['weights']),
                'character_set' => 'latin',
                'type' => 'serif',
            );
        }
        return $fonts;
    }

    /**
     * Etch: Add Adobe CSS to builder iframe
     *
     * @param array $stylesheets Existing stylesheets
     * @return array Modified stylesheets
     */
    public function etch_add_stylesheet($stylesheets) {
        $project_id = $this->get_project_id();

        if (empty($project_id)) {
            return $stylesheets;
        }

        $stylesheets[] = array(
            'id'  => 'safefonts-adobe-connect',
            'url' => sprintf(self::TYPEKIT_CDN_URL, $project_id),
        );

        return $stylesheets;
    }

}
