/**
 * SafeFonts Adobe Connect - Admin JavaScript
 */

(function($) {
    'use strict';

    var fetchedFonts = {};
    var currentProjectId = '';

    /**
     * Initialize
     */
    function init() {
        bindEvents();
        validateProjectIdInput();
    }

    /**
     * Bind events
     */
    function bindEvents() {
        $('#sfac-project-id').on('input', validateProjectIdInput);
        $('#sfac-fetch-fonts').on('click', fetchFonts);
        $('#sfac-save-fonts').on('click', saveFonts);
        $('#sfac-refresh').on('click', refreshFonts);
        $('#sfac-disconnect').on('click', disconnect);
    }

    /**
     * Validate project ID input
     */
    function validateProjectIdInput() {
        var $input = $('#sfac-project-id');
        var $button = $('#sfac-fetch-fonts');
        var value = $input.val().trim().toLowerCase();

        // Only allow lowercase letters and numbers
        $input.val(value.replace(/[^a-z0-9]/g, ''));

        // Enable button only if 7 characters
        $button.prop('disabled', value.length !== 7);
    }

    /**
     * Fetch fonts from Adobe API
     */
    function fetchFonts() {
        var projectId = $('#sfac-project-id').val().trim();

        if (projectId.length !== 7) {
            alert(sfacAdmin.strings.invalidId);
            return;
        }

        var $button = $('#sfac-fetch-fonts');
        $button.prop('disabled', true).text(sfacAdmin.strings.fetching);

        $.ajax({
            url: sfacAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sfac_fetch_fonts',
                nonce: sfacAdmin.nonce,
                project_id: projectId
            },
            success: function(response) {
                if (response.success) {
                    fetchedFonts = response.data.fonts;
                    currentProjectId = response.data.project_id;
                    renderFontsPreview(fetchedFonts, projectId);
                } else {
                    alert(response.data || sfacAdmin.strings.error);
                }
            },
            error: function() {
                alert(sfacAdmin.strings.error);
            },
            complete: function() {
                $button.prop('disabled', false).text('Fetch Fonts');
                validateProjectIdInput();
            }
        });
    }

    /**
     * Render fonts preview
     */
    function renderFontsPreview(fonts, projectId) {
        var $preview = $('#sfac-fonts-preview');
        var $list = $('#sfac-fonts-list');

        // Load Adobe CSS for preview
        if (!$('#sfac-preview-css').length) {
            $('head').append(
                '<link id="sfac-preview-css" rel="stylesheet" href="https://use.typekit.net/' + projectId + '.css">'
            );
        } else {
            $('#sfac-preview-css').attr('href', 'https://use.typekit.net/' + projectId + '.css');
        }

        var html = '';
        $.each(fonts, function(key, font) {
            html += '<div class="sfac-font-item">';
            html += '<input type="checkbox" checked data-font="' + key + '">';
            html += '<div class="sfac-font-info">';
            html += '<div class="sfac-font-name">' + escapeHtml(font.name) + '</div>';
            html += '<div class="sfac-font-weights">Weights: ' + font.weights.join(', ') + '</div>';
            html += '</div>';
            html += '<div class="sfac-font-preview" style="font-family: \'' + font.css_name + '\'">';
            html += 'The quick brown fox';
            html += '</div>';
            html += '</div>';
        });

        $list.html(html);
        $preview.show();
    }

    /**
     * Save selected fonts
     */
    function saveFonts() {
        var selectedFonts = {};

        $('#sfac-fonts-list input[type="checkbox"]:checked').each(function() {
            var fontKey = $(this).data('font');
            if (fetchedFonts[fontKey]) {
                selectedFonts[fontKey] = fetchedFonts[fontKey];
            }
        });

        if (Object.keys(selectedFonts).length === 0) {
            alert('Please select at least one font.');
            return;
        }

        var $button = $('#sfac-save-fonts');
        $button.prop('disabled', true).text(sfacAdmin.strings.saving);

        $.ajax({
            url: sfacAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sfac_save_fonts',
                nonce: sfacAdmin.nonce,
                project_id: currentProjectId,
                fonts: selectedFonts
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data || sfacAdmin.strings.error);
                    $button.prop('disabled', false).text('Connect These Fonts');
                }
            },
            error: function() {
                alert(sfacAdmin.strings.error);
                $button.prop('disabled', false).text('Connect These Fonts');
            }
        });
    }

    /**
     * Refresh fonts from API
     */
    function refreshFonts() {
        var $button = $('#sfac-refresh');
        $button.prop('disabled', true);
        $button.find('.dashicons').addClass('spin');

        // Get current project ID from data attribute (use attr to avoid jQuery caching)
        var projectId = $button.attr('data-project-id');

        $.ajax({
            url: sfacAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sfac_fetch_fonts',
                nonce: sfacAdmin.nonce,
                project_id: projectId
            },
            success: function(response) {
                if (response.success) {
                    // Save automatically with all fonts
                    $.ajax({
                        url: sfacAdmin.ajaxUrl,
                        type: 'POST',
                        data: {
                            action: 'sfac_save_fonts',
                            nonce: sfacAdmin.nonce,
                            project_id: projectId,
                            fonts: response.data.fonts
                        },
                        success: function() {
                            location.reload();
                        }
                    });
                } else {
                    alert(response.data || sfacAdmin.strings.error);
                    $button.prop('disabled', false);
                    $button.find('.dashicons').removeClass('spin');
                }
            },
            error: function() {
                alert(sfacAdmin.strings.error);
                $button.prop('disabled', false);
                $button.find('.dashicons').removeClass('spin');
            }
        });
    }

    /**
     * Disconnect Adobe Fonts
     */
    function disconnect() {
        if (!confirm(sfacAdmin.strings.confirmDisconnect)) {
            return;
        }

        var $button = $('#sfac-disconnect');
        $button.prop('disabled', true);

        $.ajax({
            url: sfacAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'sfac_disconnect',
                nonce: sfacAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data || sfacAdmin.strings.error);
                    $button.prop('disabled', false);
                }
            },
            error: function() {
                alert(sfacAdmin.strings.error);
                $button.prop('disabled', false);
            }
        });
    }

    /**
     * Escape HTML
     */
    function escapeHtml(text) {
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Initialize on document ready
    $(document).ready(init);

})(jQuery);
